<?php
/**
 * Plugin Name: Portcore GEOIP Integration
 * Description: GEOIP endpoint/key settings + GEOIP lookup helper + summary renderer.
 * Version: 1.0.0
 * Author: Portcore
 */

if (!defined('ABSPATH')) {
    exit;
}

const PC_GEOIP_OPT_ENDPOINT = 'pc_geoip_api_endpoint';
const PC_GEOIP_OPT_API_KEY = 'pc_geoip_api_key';

add_action('admin_menu', static function (): void {
    add_options_page(
        'Portcore GEOIP',
        'Portcore GEOIP',
        'manage_options',
        'portcore-geoip',
        'pc_geoip_render_settings_page'
    );
});

add_action('admin_init', static function (): void {
    register_setting('pc_geoip_settings', PC_GEOIP_OPT_ENDPOINT);
    register_setting('pc_geoip_settings', PC_GEOIP_OPT_API_KEY);
});

function pc_geoip_render_settings_page(): void
{
    if (!current_user_can('manage_options')) {
        return;
    }
    ?>
    <div class="wrap">
        <h1>Portcore GEOIP</h1>
        <form method="post" action="options.php">
            <?php settings_fields('pc_geoip_settings'); ?>
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="pc_geoip_api_endpoint">API Endpoint URL</label></th>
                    <td><input name="<?php echo esc_attr(PC_GEOIP_OPT_ENDPOINT); ?>" id="pc_geoip_api_endpoint" type="text" class="regular-text" value="<?php echo esc_attr((string)get_option(PC_GEOIP_OPT_ENDPOINT, '')); ?>" placeholder="https://geoip.space/api/check/"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="pc_geoip_api_key">API Key</label></th>
                    <td><input name="<?php echo esc_attr(PC_GEOIP_OPT_API_KEY); ?>" id="pc_geoip_api_key" type="text" class="regular-text" value="<?php echo esc_attr((string)get_option(PC_GEOIP_OPT_API_KEY, '')); ?>"></td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

function pc_geoip_normalize_ip(string $ip): string
{
    $ip = trim($ip, "[] \t\n\r\0\x0B");
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
        return $ip;
    }
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
        return strtolower($ip);
    }
    return '';
}

function pc_geoip_lookup(string $ip, array $context = []): array
{
    $ip = pc_geoip_normalize_ip($ip);
    if ($ip === '') {
        return [];
    }

    $endpoint = trim((string)($context['endpoint'] ?? get_option(PC_GEOIP_OPT_ENDPOINT, '')));
    $apiKey = trim((string)($context['api_key'] ?? get_option(PC_GEOIP_OPT_API_KEY, '')));
    if ($endpoint === '' || $apiKey === '') {
        return [];
    }

    $wpUser = wp_get_current_user();
    $userId = trim((string)($context['user_id'] ?? ''));
    if ($userId === '') {
        $userId = ($wpUser && $wpUser->ID) ? ('wpu_' . (string)$wpUser->ID) : ('wps_' . substr(sha1(wp_get_session_token()), 0, 20));
    }
    $userAgent = trim((string)($context['user_agent'] ?? ($_SERVER['HTTP_USER_AGENT'] ?? 'WordPress-PortcoreGeoip/1.0')));

    $separator = (strpos($endpoint, '?') !== false) ? '&' : '?';
    $url = rtrim($endpoint, '/') . '/' . rawurlencode($ip)
        . $separator . 'user_id=' . rawurlencode($userId)
        . '&ua=' . rawurlencode($userAgent);

    $res = wp_remote_get($url, [
        'timeout' => 12,
        'headers' => [
            'Accept' => 'application/json',
            'X-Api-Key' => $apiKey,
            'X-User-Id' => $userId,
            'User-Agent' => $userAgent,
        ],
    ]);

    if (is_wp_error($res)) {
        return [];
    }
    $code = (int)wp_remote_retrieve_response_code($res);
    $body = (string)wp_remote_retrieve_body($res);
    if ($code < 200 || $code >= 300 || $body === '') {
        return [];
    }
    $decoded = json_decode($body, true);
    return is_array($decoded) ? $decoded : [];
}

function pc_geoip_pick(array $source, array $paths, string $default = ''): string
{
    foreach ($paths as $path) {
        $cursor = $source;
        $ok = true;
        foreach (explode('.', $path) as $segment) {
            if (!is_array($cursor) || !array_key_exists($segment, $cursor)) {
                $ok = false;
                break;
            }
            $cursor = $cursor[$segment];
        }
        if ($ok && $cursor !== null && $cursor !== '') {
            return (string)$cursor;
        }
    }
    return $default;
}

function pc_geoip_summary_array(array $src, bool $showEmpty = false): array
{
    $summary = [
        'ip' => pc_geoip_pick($src, ['ip', 'query']),
        'country_code' => pc_geoip_pick($src, ['country_code', 'country.iso_code', 'countryCode']),
        'country_name' => pc_geoip_pick($src, ['country_name', 'country.name', 'country']),
        'region' => pc_geoip_pick($src, ['region', 'region_name', 'subdivisions.0.name']),
        'city' => pc_geoip_pick($src, ['city', 'city_name', 'city.name']),
        'asn' => pc_geoip_pick($src, ['asn', 'as.number', 'network.asn']),
        'org' => pc_geoip_pick($src, ['org', 'organization', 'as.organization', 'network.organization']),
        'timezone' => pc_geoip_pick($src, ['timezone', 'location.time_zone']),
        'latitude' => pc_geoip_pick($src, ['latitude', 'location.latitude']),
        'longitude' => pc_geoip_pick($src, ['longitude', 'location.longitude']),
        'is_proxy' => pc_geoip_pick($src, ['is_proxy', 'security.is_proxy']),
        'is_vpn' => pc_geoip_pick($src, ['is_vpn', 'security.is_vpn']),
        'is_tor' => pc_geoip_pick($src, ['is_tor', 'security.is_tor']),
        'risk_score' => pc_geoip_pick($src, ['risk_score', 'risk.score']),
    ];
    if ($showEmpty) {
        return $summary;
    }
    return array_filter($summary, static fn($v) => $v !== '');
}

function pc_geoip_render_summary(array $src, bool $showEmpty = false): string
{
    $summary = pc_geoip_summary_array($src, $showEmpty);
    if (empty($summary)) {
        return '<div class="pc-geoip-empty">No GEOIP fields to display.</div>';
    }

    $html = '<table class="pc-geoip-table"><tbody>';
    foreach ($summary as $k => $v) {
        $html .= '<tr><th>' . esc_html((string)$k) . '</th><td>' . esc_html((string)$v) . '</td></tr>';
    }
    $html .= '</tbody></table>';
    return $html;
}

add_shortcode('portcore_geoip_summary', static function ($atts = []): string {
    $atts = shortcode_atts([
        'data' => '',
        'show_empty' => '0',
    ], (array)$atts, 'portcore_geoip_summary');

    $data = [];
    if (is_array($atts['data'])) {
        $data = $atts['data'];
    } elseif (is_string($atts['data']) && trim($atts['data']) !== '') {
        $decoded = json_decode((string)$atts['data'], true);
        if (is_array($decoded)) {
            $data = $decoded;
        }
    }
    return pc_geoip_render_summary($data, (string)$atts['show_empty'] === '1');
});

