<?php
defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\PluginHelper;

class PortcoreGeoipHelper
{
    public static function lookup(string $ip, array $context = []): array
    {
        $ip = self::normalizeIp($ip);
        if ($ip === '') {
            return [];
        }

        $params = PluginHelper::getPlugin('system', 'portcoregeoip')->params ?? null;
        $endpoint = trim((string)($context['endpoint'] ?? ($params ? $params->get('api_endpoint', '') : '')));
        $apiKey = trim((string)($context['api_key'] ?? ($params ? $params->get('api_key', '') : '')));
        if ($endpoint === '' || $apiKey === '') {
            return [];
        }

        $user = Factory::getApplication()->getIdentity();
        $userId = trim((string)($context['user_id'] ?? ''));
        if ($userId === '') {
            $userId = ($user && (int)$user->id > 0) ? ('jmu_' . (string)$user->id) : ('jms_' . substr(sha1((string)session_id()), 0, 20));
        }
        $userAgent = trim((string)($context['user_agent'] ?? ($_SERVER['HTTP_USER_AGENT'] ?? 'Joomla-PortcoreGeoip/1.0')));

        $sep = (mb_strpos($endpoint, '?') !== false) ? '&' : '?';
        $url = rtrim($endpoint, '/') . '/' . rawurlencode($ip)
            . $sep . 'user_id=' . rawurlencode($userId)
            . '&ua=' . rawurlencode($userAgent);

        $ch = curl_init($url);
        if ($ch === false) {
            return [];
        }
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 12,
            CURLOPT_HTTPHEADER => [
                'Accept: application/json',
                'X-Api-Key: ' . $apiKey,
                'X-User-Id: ' . $userId,
                'User-Agent: ' . $userAgent,
            ],
        ]);

        $raw = curl_exec($ch);
        $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        if (!is_string($raw) || $raw === '' || $code < 200 || $code >= 300) {
            return [];
        }

        $decoded = json_decode($raw, true);
        return is_array($decoded) ? $decoded : [];
    }

    public static function getByIp(string $ip, array $context = []): array
    {
        return self::lookup($ip, $context);
    }

    public static function summaryArray(array $src, bool $showEmpty = false): array
    {
        $summary = [
            'ip' => self::pick($src, ['ip', 'query']),
            'country_code' => self::pick($src, ['country_code', 'country.iso_code', 'countryCode']),
            'country_name' => self::pick($src, ['country_name', 'country.name', 'country']),
            'region' => self::pick($src, ['region', 'region_name', 'subdivisions.0.name']),
            'city' => self::pick($src, ['city', 'city_name', 'city.name']),
            'asn' => self::pick($src, ['asn', 'as.number', 'network.asn']),
            'org' => self::pick($src, ['org', 'organization', 'as.organization', 'network.organization']),
            'timezone' => self::pick($src, ['timezone', 'location.time_zone']),
            'latitude' => self::pick($src, ['latitude', 'location.latitude']),
            'longitude' => self::pick($src, ['longitude', 'location.longitude']),
            'is_proxy' => self::pick($src, ['is_proxy', 'security.is_proxy']),
            'is_vpn' => self::pick($src, ['is_vpn', 'security.is_vpn']),
            'is_tor' => self::pick($src, ['is_tor', 'security.is_tor']),
            'risk_score' => self::pick($src, ['risk_score', 'risk.score']),
        ];

        if ($showEmpty) {
            return $summary;
        }
        return array_filter($summary, static fn($v) => $v !== '');
    }

    public static function renderSummary(array $src, bool $showEmpty = false): string
    {
        $summary = self::summaryArray($src, $showEmpty);
        if (empty($summary)) {
            return '<div class="pc-geoip-empty">No GEOIP fields to display.</div>';
        }

        $html = '<table class="pc-geoip-table"><tbody>';
        foreach ($summary as $k => $v) {
            $html .= '<tr><th>' . htmlspecialchars((string)$k, ENT_QUOTES) . '</th><td>' . htmlspecialchars((string)$v, ENT_QUOTES) . '</td></tr>';
        }
        $html .= '</tbody></table>';
        return $html;
    }

    protected static function normalizeIp(string $ip): string
    {
        $ip = trim($ip, "[] \t\n\r\0\x0B");
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            return $ip;
        }
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            return strtolower($ip);
        }
        return '';
    }

    protected static function pick(array $source, array $paths, string $default = ''): string
    {
        foreach ($paths as $path) {
            $cursor = $source;
            $ok = true;
            foreach (explode('.', $path) as $segment) {
                if (!is_array($cursor) || !array_key_exists($segment, $cursor)) {
                    $ok = false;
                    break;
                }
                $cursor = $cursor[$segment];
            }
            if ($ok && $cursor !== null && $cursor !== '') {
                return (string)$cursor;
            }
        }
        return $default;
    }
}

