<?php
namespace Portcore\Geoip;

class Client
{
    /**
     * Lookup GEOIP data by explicit IP.
     *
     * @param string $ip
     * @param array<string, string|int> $context Optional context:
     *  - user_id
     *  - user_agent
     *  - endpoint
     *  - api_key
     * @return array<string, mixed>
     */
    public static function lookup(string $ip, array $context = []): array
    {
        $ip = self::normalizeIp($ip);
        if ($ip === '') {
            return [];
        }

        $ctx = self::buildContext($context);
        $endpoint = $ctx['endpoint'];
        $apiKey = $ctx['api_key'];
        $userId = $ctx['user_id'];
        $userAgent = $ctx['user_agent'];

        if ($endpoint === '' || $apiKey === '') {
            return [];
        }

        $separator = (mb_strpos($endpoint, '?') !== false) ? '&' : '?';
        $url = rtrim($endpoint, '/') . '/' . rawurlencode($ip)
            . $separator . 'user_id=' . rawurlencode($userId)
            . '&ua=' . rawurlencode($userAgent);

        $ch = curl_init($url);
        if ($ch === false) {
            return [];
        }

        $headers = [
            'Accept: application/json',
            'X-Api-Key: ' . $apiKey,
            'X-User-Id: ' . $userId,
            'User-Agent: ' . $userAgent,
        ];

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 12,
            CURLOPT_HTTPHEADER => $headers,
        ]);

        $raw = curl_exec($ch);
        $httpCode = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if (!is_string($raw) || $raw === '' || $httpCode < 200 || $httpCode >= 300) {
            return [];
        }

        $decoded = json_decode($raw, true);
        return is_array($decoded) ? $decoded : [];
    }

    /**
     * Alias method with explicit semantic naming.
     *
     * @param string $ip
     * @param array<string, string|int> $context
     * @return array<string, mixed>
     */
    public static function getByIp(string $ip, array $context = []): array
    {
        return self::lookup($ip, $context);
    }

    protected static function normalizeIp(string $ip): string
    {
        $ip = trim($ip);
        if ($ip === '') {
            return '';
        }
        $ip = trim($ip, "[] \t\n\r\0\x0B");
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            return $ip;
        }
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            return strtolower($ip);
        }
        return '';
    }

    /**
     * @param array<string, string|int> $context
     * @return array{endpoint:string,api_key:string,user_id:string,user_agent:string}
     */
    protected static function buildContext(array $context): array
    {
        $endpoint = trim((string)($context['endpoint'] ?? Settings::getEndpoint()));
        $apiKey = trim((string)($context['api_key'] ?? Settings::getApiKey()));

        $userId = trim((string)($context['user_id'] ?? ''));
        if ($userId === '' && isset($GLOBALS['USER']) && is_object($GLOBALS['USER']) && method_exists($GLOBALS['USER'], 'GetID')) {
            $uid = (string)$GLOBALS['USER']->GetID();
            if ($uid !== '') {
                $userId = 'bxu_' . $uid;
            }
        }
        if ($userId === '') {
            $userId = 'bxs_' . substr(sha1((string)session_id()), 0, 20);
        }

        $userAgent = trim((string)($context['user_agent'] ?? ($_SERVER['HTTP_USER_AGENT'] ?? '')));
        if ($userAgent === '') {
            $userAgent = 'Bitrix-PortcoreGeoip/1.0';
        }

        return [
            'endpoint' => $endpoint,
            'api_key' => $apiKey,
            'user_id' => $userId,
            'user_agent' => $userAgent,
        ];
    }
}
